var Core = require('../../NGCore/Client/Core').Core;
var GL2  = require('../../NGCore/Client/GL2').GL2;
var utils = require('../../DnLib/Dn/utils').utils;


exports.Trajectory = Core.MessageListener.subclass({

	initialize: function(touchEmitter, length, colorHead, colorTail) {
		//-----
		this._emitter = touchEmitter;
		this._maxLength = length || 20;
		this._colorHead = colorHead || [1, 1, 0];
		this._colorTail = colorTail || [1, 0, 0];
		this._bladePower = 1.0; // 1.0 が MAX. 刀を折り返すたびに威力が減って軌跡が細くなる
		this._vertexList = [];
		this._isMoving   = false;

		//-----
		this._emitter.addListener( this, this.onTouch );

		//-----
		this._node = new GL2.Node();
		
		//-----
		var p = new GL2.Primitive();
		p.setType( GL2.Primitive.Type.TriangleStrip );
		for (var i=0; i < this._maxLength * 2; i++) {
			var v = new GL2.Primitive.Vertex([0, 0], [0, 0], [1, 0, 0]);
			p.pushVertex(v);
		}
		this._prim = p;
		this._node.addChild( this._prim );
		this._node.setDepth( 1 );
		
		//-----
		Core.UpdateEmitter.addListener( this, this.onUpdate );
	},
	
	//--------------------------------------------------------------------------
	destroy: function() {
		Core.UpdateEmitter.removeListener(this);
		this._emitter.removeListener(this);
		this._prim.destroy();
	},
	
	//--------------------------------------------------------------------------

	getVertexList: function() {
		return this._vertexList;
	},

	getNode: function() {
		return this._node;
	},
	
	//--------------------------------------------------------------------------
	onUpdate: function( delta ) {
		
		var v = this._vertexList;
		var p = this._prim;
		if (v.length >= 2) {
			p.setVisible( true );
			
			var bladeWidth = 10 * this._bladePower;
			for (var i=0;  i < v.length - 1;  i++) {
				var grad = 1 - (i / v.length);
				var r = (this._colorHead[0] * grad) + (this._colorTail[0] * (1-grad));
				var g = (this._colorHead[1] * grad) + (this._colorTail[1] * (1-grad));
				var b = (this._colorHead[2] * grad) + (this._colorTail[2] * (1-grad));
				
				var w  = (i < 4) ? (i * 0.22) : g;
				var bw = w * bladeWidth;
				
				var vx  = v[i  ].x;
				var vpx = v[i+1].x;
				var vy  = v[i  ].y;
				var vpy = v[i+1].y;
				
				//----- 進行方向と直角に軌跡の太さの方向を算出
				if (! v[i].dx) {
					var theta = Math.atan2( vy - vpy, vx - vpx ) * 180 / Math.PI;
					v[i].dx = utils.cos( theta + 90 );
					v[i].dy = utils.sin( theta + 90 );
				}
				
				p.setVertex( i*2+0, new GL2.Primitive.Vertex(
					[v[i].x - v[i].dx * bw, v[i].y - v[i].dy * bw], [0, 0], [r, g, b]));
				p.setVertex( i*2+1, new GL2.Primitive.Vertex(
					[v[i].x + v[i].dx * bw, v[i].y + v[i].dy * bw], [0, 0], [r, g, b]));
			}

			//----- 余った頂点は末尾にまとめる
			var lastVertex = new GL2.Primitive.Vertex(
				[v[ v.length - 1 ].x, v[ v.length - 1 ].y], [0, 0], [0, 0, 0]);
			for (var i = v.length - 1;  i < this._maxLength;  i++) {
 				p.setVertex( i*2+0, lastVertex );
 				p.setVertex( i*2+1, lastVertex );
			}

		} else {
			p.setVisible(false);
		}
		
		if (v.length > 0  && !this._isMoving) {
			v.pop();
		} else {
			this._isMoving = false;
		}

	},
	
	//--------------------------------------------------------------------------
	onTouch: function( touch ) {
		
		var x = touch.getPosition().getX();
		var y = touch.getPosition().getY();

		this._isMoving = false;

		switch (touch.getAction()) {
		case touch.Action.Move:
			this._vertexList.unshift( {x:x, y:y} );
			this._isMoving = true;
			break;

		case touch.Action.Start:
		case touch.Action.End:
		default:
			break;
		}
		
		if (this._vertexList.length > this._maxLength) {
			this._vertexList.pop();
		}

		return touch.getAction() === touch.Action.Start;
	}

});
