var Core = require('../../NGCore/Client/Core').Core;
var GL2  = require('../../NGCore/Client/GL2').GL2;
var utils = require('../../DnLib/Dn/utils').utils;


//========================================================================================
exports.Trajectory = Core.MessageListener.subclass({
	
	// @param {dn.globalTouchEmitter} touchEmitter
	// @param {Number} length: the number of the vertices of the primitive. (default: 20)
	//--------------------------------------------------------------------------
	initialize: function(touchEmitter, length, colorHead, colorTail) {

		this.node = new GL2.Node();
		
		this.emitter = touchEmitter;
		if (this.emitter) {
			this.emitter.addListener( this, null, this.onTouch );
		}
		this.maxLength = length || 20;
		this.colorHead = colorHead || [1, 1, 0];
		this.colorTail = colorTail || [1, 0, 0];
		
		this.vertexList = [];
		this.moveCount  = 0;
		this.isMoving   = false;
		
		var p = new GL2.Primitive();
		p.setType( GL2.Primitive.Type.TriangleStrip );
		for (var i=0;  i < this.maxLength * 2;  i++) {
			p.pushVertex( new GL2.Primitive.Vertex([0, 0], [0, 0], [1, 0, 0]) );
		}
		this.node.prim = p;
		this.node.addChild( this.node.prim );
		this.node.setDepth( 1 );
		this.tapCount = 0;
		
		this.x = 0;
		this.y = 0;
		this.prevTouchId = null;
		
		this.bladePower       = 1.0 ; // 1.0 が MAX. 刀を折り返すたびに威力が減って軌跡が細くなる
		this.bladePowerMin    = 0.1 ; // 最後は細い線で止まる

		Core.UpdateEmitter.addListener( this, this.onUpdate );
	},
	
	//--------------------------------------------------------------------------
	destroy: function() {
		if (this.emitter) {
			this.emitter.removeListener( this );
		}
	},
	
	//--------------------------------------------------------------------------
	getVertexList: function() {
		return this.vertexList;
	},

	getLength: function() {
		return this.vertexList.length;
	},
	
	//--------------------------------------------------------------------------
	onUpdate: function( delta ) {
		
		var v = this.vertexList;
		var p = this.node.prim;
		if (v.length >= 2) {
			p.setVisible( true );
			
			var bladeWidth = 10 * this.bladePower;
			for (var i=0;  i < v.length - 1;  i++) {
				var grad = 1 - (i / v.length);
				var r = (this.colorHead[0] * grad) + (this.colorTail[0] * (1-grad));
				var g = (this.colorHead[1] * grad) + (this.colorTail[1] * (1-grad));
				var b = (this.colorHead[2] * grad) + (this.colorTail[2] * (1-grad));
				
				var w  = (i < 4) ? (i * 0.22) : g;
				var bw = w * bladeWidth;
				
				var vx  = v[i  ].x;
				var vpx = v[i+1].x;
				var vy  = v[i  ].y;
				var vpy = v[i+1].y;
				
				//----- 進行方向と直角に軌跡の太さの方向を算出
				if (! v[i].dx) {
					var theta = Math.atan2( vy - vpy, vx - vpx ) * 180 / Math.PI;
					v[i].dx = utils.cos( theta + 90 );
					v[i].dy = utils.sin( theta + 90 );
				}
				
				p.setVertex( i*2+0, new GL2.Primitive.Vertex(
					[v[i].x - v[i].dx * bw, v[i].y - v[i].dy * bw], [0, 0], [r, g, b]
				));
				p.setVertex( i*2+1, new GL2.Primitive.Vertex(
					[v[i].x + v[i].dx * bw, v[i].y + v[i].dy * bw], [0, 0], [r, g, b]
				));
			}

			//----- 余った頂点は末尾にまとめる
			var lastVertex = new GL2.Primitive.Vertex(
				[v[ v.length - 1 ].x, v[ v.length - 1 ].y], [0, 0], [0, 0, 0]
			);
			for (var i = v.length - 1;  i < this.maxLength;  i++) {
 				p.setVertex( i*2+0, lastVertex );
 				p.setVertex( i*2+1, lastVertex );
			}
		}
		else {
			p.setVisible( false );
		}
		
		if (v.length > 0  &&  !this.isMoving) {
			v.pop();
		} else {
			this.isMoving = false;
		}

	},
	
	//--------------------------------------------------------------------------
	onTouch: function( touch ) {
		
		//ToDo: これダサいので何とかしたい
		var scale = 320 / Core.Capabilities.getScreenWidth();
		var x = touch.getPosition().getX() * scale;
		var y = touch.getPosition().getY() * scale;
		this.x = x;
		this.y = y;
		this.isMoving   = false;
		this.isTouching = true;
		
		var vList = this.vertexList;
		
		switch (touch.getAction()) {
			//----------------------------------------
		case touch.Action.Start:
			//this.vertexList = [ {x:x, y:y} ];
			//----- GALAXY S とか NEXUS S は５本指まで対応しているらしいが、３本指以上だと START と END の
			//----- 数が合わなくなる場合があるみたいなので仕方なく２本までをカウントしてる。
			//----- ２本以上押したときに刀を判定させなくするため。
			this.tapCount++;
			if (this.tapCount > 2) { this.tapCount = 2; }
			break;
			
			//----------------------------------------
		case touch.Action.End:
			this.tapCount--;
			if (this.tapCount < 0) { this.tapCount = 0; }
			if (this.prevTouchId === touch.getId()) { this.prevTouchId = null; }
			this.moveCount = 0;
			break;
			
			//----------------------------------------
		case touch.Action.Move:
			//----- 前回と違う ID なら指が２本以上触れている
			//----- ３本以上の START と END の整合性が保証されないのでこんなチェックをしてる
			if (this.prevTouchId  &&  this.prevTouchId !== touch.getId()) {
				if (this.tapCount < 2) { this.tapCount = 2; }
			}
			this.prevTouchId = touch.getId();
			
			if (this.tapCount > 1) {
				vList = [ {x:x, y:y} ];
			} else {
				vList.unshift( {x:x, y:y} );
				this.isMoving = true;
				++this.moveCount;
			}
			break;
		}
		
		if (vList.length > this.maxLength) {
			vList.pop();
		}
	}

});
